function [nPassNodeInLv, nNewPassNodeInLv, nCumNodeInLv] = countIOPassNode(network, nLevel)
% COUNTIOPASSNODE count number of passed nodes, new nodes within pathway in specific levels
%
%   [nPassNodeInLv, nNewPassNodeInLv, nCumNodeInLv] = countIOPassNode(network, nLevel)
%   This function count number of passed nodes, new nodes and cumulative nodes
%   within input-ouput pair pathway in different levels.
%   NOTE: This function may take several hours to analyse.
%
%   Input:
%       network: input network
%       nLevel: max level to analyse
%
%   Output:
%       nPassNodeInLv: cell array containing number of passed nodes in different
%           propation levels
%       nNewPassNodeInLv: cell array containing number of new nodes in different
%           propation levels
%       nCumNodeInLv: cell array containing number of cumulative nodes in
%           different propation levels

%   ---------
%   Yen-Nan Lin, NTHU, 2010-2014, Matlab 2012a

connectMat = network.matrix;
connectMat = single(full(connectMat));
[nRow, nCol] = size(connectMat);
nInput = network.inputNumber;
nOutput = network.outputNumber;
inputIx = 1:nInput;
outputIx = (nInput + 1):(nInput + nOutput);

connectIxEachCol = cell(1, nCol);
for iCol = 1:nCol
    connectIxEachCol{iCol} = single(find(connectMat(:, iCol) > 0));
end

preLevelMat = eye(nRow, nCol);
preLvPassNode = cell(nRow, nCol);
cumNodeInLv{1} = preLvPassNode;

for iLevel = 1:nLevel
    thisLvPassNode = cell(nRow, nCol);
    for jRow = 1:nRow
        sourceIxEachRow{jRow} = single(find(preLevelMat(jRow, :) > 0));
    end

    for jRow = inputIx
        for kCol = 1:nCol
            pathList = intersect(sourceIxEachRow{jRow}, connectIxEachCol{kCol});
            if ~isempty(pathList)
                pathList = union(pathList, kCol);
            end

            thisLvPassNode{jRow, kCol} = ...
                union(pathList, [preLvPassNode{jRow, pathList}]);
            if iLevel == 1
                newPassNodeInLv{iLevel}{jRow, kCol} = setdiff( ...
                    thisLvPassNode{jRow, kCol}, cumNodeInLv{1}{jRow, kCol});
                cumNodeInLv{iLevel}{jRow, kCol} = union( ...
                    cumNodeInLv{1}{jRow, kCol}, thisLvPassNode{jRow, kCol});
            else
                newPassNodeInLv{iLevel}{jRow, kCol} = setdiff( ...
                    thisLvPassNode{jRow, kCol}, ...
                    cumNodeInLv{iLevel - 1}{jRow, kCol});
                cumNodeInLv{iLevel}{jRow, kCol} = union( ...
                    cumNodeInLv{iLevel - 1}{jRow, kCol}, ...
                    thisLvPassNode{jRow, kCol});
            end
        end
    end
    passNodeInLv{iLevel} = thisLvPassNode;
    preLvPassNode = thisLvPassNode;
    preLevelMat = preLevelMat * connectMat;

    nPassNodeInLv{iLevel} = ...
        single(cellfun(@numel, passNodeInLv{iLevel}(inputIx, outputIx)));
    nNewPassNodeInLv{iLevel} = ...
        single(cellfun(@numel, newPassNodeInLv{iLevel}(inputIx, outputIx)));
    nCumNodeInLv{iLevel} = ...
        single(cellfun(@numel, cumNodeInLv{iLevel}(inputIx, outputIx)));
end
end
