% This script analyze C. Elegans neural network, Drosophila's cnetral complex
% neural network, and several theoretical networks generated based on C. Elegans
% neural network.
%
% The standard process of analysis:
% # load neural network
% # analyse neural network
% # create theoretical networks based on the loaded neural network
% # analyse theoretical networks
% # compute the average of realisation results

%   ---------
%   Yen-Nan Lin, NTHU, 2010-2014, Matlab 2012a

close all;
clear all;
goIncludeToolbox;

% parameter for analysis
parameter.MAX_LEVEL = 8;
parameter.REPEAT_TIMES = 100;
parameter.STRIDE_SEED_NUMBER = 10000;
parameter.REWIRE_PROB_RING = 0.3;
parameter.ANALYSIS_RECURRENTION = false;

% rewiring probabilites of SW networks
REWIRE = [0.01 0.03 0.05 0.1 0.15 0.20 0.4 0.5];

% load parameter struct into base workspace
loadStruct(parameter, 'base');

% C. Elegans network analysis
ceNetwork = bioNetwork('./CE_ExpData/C_elegans_Matrix.mat', 'mat');
ceNetwork = readLabel(ceNetwork, './CE_ExpData/C_elegans_Label.txt');
ceNetwork = setIO(ceNetwork, ceNetwork.inputList, ceNetwork.outputList);
rand('seed', 0);
ceNetwork = levelAnalysis(ceNetwork, MAX_LEVEL);

% generate theoretical networks based on C. Elegans neural network
% and analyze the networks
[meanNetworks, allNetworks, stdNetworks] = ...
    makeNetAndAnalysis(ceNetwork, parameter);
% load meanNetworks and allNetworks to base workspace
loadStruct(meanNetworks, 'base');
loadStruct(allNetworks, 'base');
loadStruct(stdNetworks, 'base');
clear meanNetworks allNetworks stdNetworks

% central complex network analysis
cxNetwork = bioNetwork('./PcbExpData/CXGraph_AllRevised.mat', 'mat');
cxNetwork = readLabel(cxNetwork, './PcbExpData/CXLabel_AllRevised.txt');
cxNetwork = setIO(cxNetwork, cxNetwork.inputList, cxNetwork.outputList);
rand('seed', 0);
cxNetwork = levelAnalysis(cxNetwork, MAX_LEVEL);

% analyze SW netwroks with rewiring probabilites from 0.01 to 0.5
tmpMeanNumEdge = full(mean(sum(ceNetwork.matrix')));
% change rewire probabilites of SW network
for iRewire = 1:numel(REWIRE)
    smallWireNet{iRewire} = cell(1, REPEAT_TIMES);
    for jNet = 1:REPEAT_TIMES
        seedNumber = jNet * STRIDE_SEED_NUMBER;
        smallWireNet{iRewire}{jNet} = smallWorldNetwork(ceNetwork.size, ...
            REWIRE(iRewire), seedNumber, tmpMeanNumEdge);
        smallWireNet{iRewire}{jNet} = setIO(smallWireNet{iRewire}{jNet}, ...
            ceNetwork.inputNumber, ceNetwork.outputNumber);
        smallWireNet{iRewire}{jNet} = ...
            levelAnalysis(smallWireNet{iRewire}{jNet}, MAX_LEVEL);
    end
    % compute mean and std result of analysis
    meanSmallNet{iRewire} = meanFieldInCells(smallWireNet{iRewire});
    stdSmallNet{iRewire} = stdFieldInCells(smallWireNet{iRewire});
    % remove individul network data to save memory usage
    smallWireNet{iRewire} = [];
end
clear smallWireNet

% randomly select I/O neurons of C. Elegans neural network
for iNet = 1:REPEAT_TIMES
    seedNumber = iNet * STRIDE_SEED_NUMBER;
    rand('seed', seedNumber);
    randIoCeNet{iNet} = ...
        bioNetwork('./CE_ExpData/C_elegans_Matrix.mat', 'mat');
    randIoCeNet{iNet} = setIO(randIoCeNet{iNet}, ...
        ceNetwork.inputNumber, ceNetwork.outputNumber);
    randIoCeNet{iNet} = levelAnalysis(randIoCeNet{iNet}, MAX_LEVEL);
end
meanRandIoCeNet = meanFieldInCells(randIoCeNet);
stdRandIoCeNet = stdFieldInCells(randIoCeNet);

% select I/O neurons in separated modules
for iNet = 1:REPEAT_TIMES
    seedNumber = iNet * STRIDE_SEED_NUMBER;
    rand('seed', seedNumber);
    sepIoCeNet{iNet} = ...
        bioNetwork('./CE_ExpData/C_elegans_Matrix.mat', 'mat');
    sepIoCeNet{iNet} = ...
        readLabel(sepIoCeNet{iNet}, './CE_ExpData/C_elegans_Label.txt');
    % select I/O neuron by modules
    sepIoCeNet{iNet} = setIoByModule(sepIoCeNet{iNet}, ...
        numel(sepIoCeNet{iNet}.inputList), numel(sepIoCeNet{iNet}.outputList));
    sepIoCeNet{iNet} = levelAnalysis(sepIoCeNet{iNet}, MAX_LEVEL);
end
meanSepIoCeNet = meanFieldInCells(sepIoCeNet);
stdSepIoCeNet = stdFieldInCells(sepIoCeNet);

% Reverse I/O neurons of C. Elegans network
revIoCeNet = bioNetwork('./CE_ExpData/C_elegans_Matrix.mat', 'mat');
revIoCeNet = readLabel(revIoCeNet, './CE_ExpData/C_elegans_Label.txt');
revIoCeNet = ...
    setIO(revIoCeNet, revIoCeNet.outputList, revIoCeNet.inputList);
rand('seed', 0);
revIoCeNet = levelAnalysis(revIoCeNet, MAX_LEVEL);