% This script analyze C. Elegans neural network, Drosophila's cnetral complex
% neural network, and several theoretical networks generated based on C. Elegans
% neural network.
% NOTE: Due to count number of new nodes, this script may take several hours
% to analysis.
%
% The standard process of analysis:
% # load neural network
% # analyse neural network
% # create theoretical networks based on the loaded neural network
% # analyse theoretical networks
% # compute the average of realisation results

%   ---------
%   Yen-Nan Lin, NTHU, 2010-2014, Matlab 2012a

close all;
clear all;
goIncludeToolbox;

% parameter for analysis
parameter.MAX_LEVEL = 8;
parameter.REPEAT_TIMES = 100;
parameter.STRIDE_SEED_NUMBER = 10000;
parameter.REWIRE_PROB_RING = 0.3;
parameter.ANALYZE_RECURRENT = true;

% load parameter struct into base workspace
loadStruct(parameter, 'base');

% C. Elegans network analysis
ceNetwork = bioNetwork('./CE_ExpData/C_elegans_Matrix.mat', 'mat');
ceNetwork = readLabel(ceNetwork, './CE_ExpData/C_elegans_Label.txt');
ceNetwork = setIO(ceNetwork, ceNetwork.inputList, ceNetwork.outputList);
rand('seed', 0);
ceNetwork = levelAnalysis(ceNetwork, MAX_LEVEL, ANALYZE_RECURRENT);


% compute number of edges information of ceNetwork
ceNetwork.NumConnect = full(sum(sum(ceNetwork.matrix)));
ceNetwork.meanNumEdge = full(mean(sum(ceNetwork.matrix')));

% generate theoretical networks based on C. Elegans neural network
% and analyze the networks
% random network analysis
randProb = ceNetwork.NumConnect / (ceNetwork.size^2 - ceNetwork.size);
randNetwork = cell(1, REPEAT_TIMES);
for iTime = 1:REPEAT_TIMES
    seedNumber = iTime * STRIDE_SEED_NUMBER;
    randNetwork{iTime} = randomNetwork(ceNetwork.size, randProb, seedNumber);
    randNetwork{iTime} = setIO(randNetwork{iTime}, ...
        ceNetwork.inputNumber, ceNetwork.outputNumber);
    randNetwork{iTime} = ...
        levelAnalysis(randNetwork{iTime}, MAX_LEVEL, ANALYZE_RECURRENT);
end
% compute mean result of analysis
meanRandNet = meanFieldInCells(randNetwork);
stdRandNet = stdFieldInCells(randNetwork);

% ring type small world network analysis
smallRingNet = cell(1, REPEAT_TIMES);
for iTime = 1:REPEAT_TIMES
    seedNumber = iTime * STRIDE_SEED_NUMBER;
    smallRingNet{iTime} = smallWorldNetwork(ceNetwork.size, ...
        REWIRE_PROB_RING, seedNumber, ceNetwork.meanNumEdge);
    smallRingNet{iTime} = setIO(smallRingNet{iTime}, ...
        ceNetwork.inputNumber, ceNetwork.outputNumber);
    smallRingNet{iTime} = ...
        levelAnalysis(smallRingNet{iTime}, MAX_LEVEL, ANALYZE_RECURRENT);
end
% compute mean result of analysis
meanRingNet = meanFieldInCells(smallRingNet);
stdRingNet = stdFieldInCells(smallRingNet);

% central complex network analysis
cxNetwork = bioNetwork('./PcbExpData/CXGraph_AllRevised.mat', 'mat');
cxNetwork = readLabel(cxNetwork, './PcbExpData/CXLabel_AllRevised.txt');
cxNetwork = setIO(cxNetwork, cxNetwork.inputList, cxNetwork.outputList);
rand('seed', 0);
cxNetwork = levelAnalysis(cxNetwork, MAX_LEVEL, ANALYZE_RECURRENT);