% This script produces the figure 4. It shows the distribution of horizontal
% propagation of input nodes in C. Elegans neural network, small-world network
% random network, Drosophila's CX network in level 0-3.

%   ---------
%   Yen-Nan Lin, NTHU, 2010-2014, Matlab 2012a

close all;

% parameter of plot
FIG_SIZE = [0 0 1000 800];
MAX_PLOT_LEVEL = 4;
NUM_BINS = [5 10 10 10]; % number of bins in each level
SMALL_VALUE = 10^(-10); % to avoid log(0)
lineWidth = 3;
markerSize = 8;
markerScale = 1.6;

% setting of figure
figure('outerPosition', FIG_SIZE, 'color', 'w');
panelObj = panel('defer');
panelObj.pack(2, 2);
panelObj.fontsize = 15;
panelObj.fontname = 'Arial';
panelObj.fontweight = 'bold';
panelObj.margin = [20 15 5 5];
panelObj.de.margin = [15 15 5 5];

for iLevel = 1:MAX_PLOT_LEVEL
    [colIx, rowIx] = ind2sub([2, 2], iLevel);
    panelObj(rowIx, colIx).select();

    % collect horizontal propagation of each input nodes
    ceHorzProp = rowPathRatio(ceNetwork.LevelMatrix{iLevel});
    cxHorzProp = rowPathRatio(cxNetwork.LevelMatrix{iLevel});
    randHorzProp = [];
    smallHorzProp = [];
    for jNet = 1:numel(randNetwork)
        tmpHorzProp = rowPathRatio(randNetwork{jNet}.LevelMatrix{iLevel});
        randHorzProp = vertcat(randHorzProp, tmpHorzProp);
        tmpHorzProp = rowPathRatio(smallRingNet{jNet}.LevelMatrix{iLevel});
        smallHorzProp = vertcat(smallHorzProp, tmpHorzProp);
    end

    % compute probability density of each network
    [ceProbDens, ceXOut] = probDensity(ceHorzProp, NUM_BINS(iLevel));
    [cxProbDens, pcbXOut] = probDensity(cxHorzProp, NUM_BINS(iLevel));
    [randProbDens, randXOut] = probDensity(randHorzProp, NUM_BINS(iLevel));
    [smallProbDens, smallXOut] = probDensity(smallHorzProp, NUM_BINS(iLevel));
    % adding small value to prevent log(0)
    ceProbDens = ceProbDens + SMALL_VALUE;
    cxProbDens = cxProbDens + SMALL_VALUE;
    randProbDens = randProbDens + SMALL_VALUE;
    smallProbDens = smallProbDens + SMALL_VALUE;

    % plot probability density of each network
    hold on;
    semilogy(ceXOut, ceProbDens, 'Marker', netMarker('CE'), 'Color', ...
        netColor('CE'), 'MarkerFaceColor', netMarkerFace('CE'), ...
        'LineWidth', lineWidth, 'MarkerSize', markerSize);
    semilogy(pcbXOut, cxProbDens, 'Marker', netMarker('CX'), 'Color', ...
        netColor('CX'), 'MarkerFaceColor', netMarkerFace('CX'), ...
        'LineWidth', lineWidth, 'MarkerSize', markerSize * markerScale);
    semilogy(randXOut, randProbDens, 'Marker', netMarker('ER'), 'Color', ...
        netColor('ER'), 'MarkerFaceColor', netMarkerFace('ER'), ...
        'LineWidth', lineWidth, 'MarkerSize', markerSize * markerScale);
    semilogy(smallXOut, smallProbDens, 'Marker', netMarker('SW'), 'Color', ...
        netColor('SW'), 'MarkerFaceColor', netMarkerFace('SW'), ...
        'LineWidth', lineWidth, 'MarkerSize', markerSize);

    % setting plot
    set(gca, 'yscale', 'log', 'lineWidth', 2, 'yTick', [10^-2, 1, 10^2]);
    xlim([0, 1]);
    ylim([10^(-3), 10^2]);
    if iLevel == 1
        legend('CE', 'CX', 'ER', 'SW');
        legend('location', 'NorthEast');
        hL = legend('boxoff');
        set(hL, 'fontsize', 12);
    end
end

panelObj.xlabel('Horizontal propagation');
panelObj.ylabel('Probability density');
ylabelH = findobj('string', 'Probability density');
panelObj.refresh();
set(ylabelH, 'position', [0.1 1 0]);