% This script produces the figure 5a. It plot the vertical propagation versus
% propagation levels in C. Elegans neural network, small-world network, random
% network, regular network and Drosophila's CX network. The curves are based on
% fitting Gompertz function. The errorbar is standard deviation.

%   ---------
%   Yen-Nan Lin, NTHU, 2010-2014, Matlab 2012a

close all;

% plot setting
FIG_SIZE = [0 0 1200 600];
lineWidth = 3;
markerSize = 10;
markerScale = 1.5;

% fitting with Gompertz function
cfit.CE = fitGompertz(0:(MAX_LEVEL - 2), ceNetwork.VertProp);
cfit.Rand = fitGompertz(0:(MAX_LEVEL - 2), meanRandNet.VertProp);
cfit.Regular = fitGompertz(0:(MAX_LEVEL - 2), meanRegularNet.VertProp);
cfit.Ring = fitGompertz(0:(MAX_LEVEL - 2), meanRingNet.VertProp);
cfit.CX = fitGompertz(0:(MAX_LEVEL - 2), cxNetwork.VertProp);

% save cfit object to each network
ceNetwork.cfitVertProp = cfit.CE;
meanRandNet.cfitVertProp = cfit.Rand;
meanRegularNet.cfitVertProp = cfit.Regular;
meanRingNet.cfitVertProp = cfit.Ring;
cxNetwork.cfitVertProp = cfit.CX;

% plot marker and errorbar of each networks
figure('name', 'Vertical Propagation', 'outerPosition', FIG_SIZE, 'color', 'w');
hold on;
tmpP = linspace(0, MAX_LEVEL - 2, 100);
plot(0:(MAX_LEVEL - 2), ceNetwork.VertProp, 'Marker', netMarker('CE'), ...
    'Color', netColor('CE'), 'MarkerSize', markerSize, ...
    'MarkerFaceColor', netMarkerFace('CE'), ...
    'LineWidth', lineWidth, 'LineStyle', 'none');
errorbar(0:(MAX_LEVEL - 2), meanRandNet.VertProp, ...
    stdRandNet.VertProp, 'Marker', netMarker('ER'), ...
    'Color', netColor('ER'), 'MarkerSize', markerSize * markerScale, ...
    'MarkerFaceColor', netMarkerFace('ER'), ...
    'LineWidth', lineWidth, 'LineStyle', 'none');
errorbar(0:(MAX_LEVEL - 2), meanRegularNet.VertProp, ...
    stdRegularNet.VertProp, 'Marker', netMarker('RL'), ...
    'Color', netColor('RL'), 'MarkerSize', markerSize, ...
    'MarkerFaceColor', netMarkerFace('RL'), ...
    'LineWidth', lineWidth, 'LineStyle', 'none');
errorbar(0:(MAX_LEVEL - 2), meanRingNet.VertProp, ...
    stdRingNet.VertProp, 'Marker', netMarker('SW'), ...
    'Color', netColor('SW'), 'MarkerSize', markerSize, ...
    'MarkerFaceColor', netMarkerFace('SW'), ...
    'LineWidth', lineWidth, 'LineStyle', 'none');
plot(0:(MAX_LEVEL - 2), cxNetwork.VertProp, 'Marker', netMarker('CX'), ...
    'Color', netColor('CX'), 'MarkerSize', markerSize * markerScale, ...
    'MarkerFaceColor', netMarkerFace('CX'), ...
    'LineWidth', lineWidth, 'LineStyle', 'none');

% plot fitting curve of each network
plot(tmpP, cfit.CE(tmpP), 'Color', netColor('CE'), 'LineWidth', 4);
plot(tmpP, cfit.Rand(tmpP), 'Color', netColor('ER'), 'LineWidth', 4);
plot(tmpP, cfit.Regular(tmpP), 'Color', netColor('RL'), 'LineWidth', 4);
plot(tmpP, cfit.Ring(tmpP), 'Color', netColor('SW'), 'LineWidth', 4);
plot(tmpP, cfit.CX(tmpP), 'Color', netColor('CX'), 'LineWidth', 4);
plot([2 2], [0 cxNetwork.cfitVertProp(2)], '--k', 'lineWidth', 3);
set(gca, 'FontSize', 18, 'lineWidth', 4, 'FontWeight', 'bold');

hL = legend('CE', 'ER', 'RL', 'SW', 'CX', ...
    'location', 'southEast');
legend(gca, 'boxoff');
% to show the correct linewidth of errorbar's symbol due to matlab's bug
set(findobj(hL, 'type', 'line'), 'LineWidth', lineWidth);

xlim([0 4]);
ylim([0 1.1]);
xlabel('Propagation level');
ylabel('Vertical propagation');
set(gca, 'xTick', [0 1 2 3 4], 'yTick', [0 0.4 0.8 1]);
box off;