function [ network ] = levelAnalysis( network, maxLevel, doNewNode )
% LEVELANALYSIS do horizontal, vertical analyses, and so on.
%
%   [ network ] = levelAnalysis( network, maxLevel, doNewNode )
%   This function do several network analyses. It rearrage position of nodes
%   by modules (nodes in the same modules would be close together), count number
%   of new nodes within pathway in different propagation level (if doNewNode is
%   true), compute pathway number between input-output pairs in different
%   propagation levels, generate channel connectivity matrix in different
%   levels, compute vertical and horizontal propagation, compute participation
%   coefficient by the module sorted before.
%
%   sortByModule() -> reorder nodes by arraging the same module node together
%   countIOPassNode() -> compute number of new nodes in levels
%   computePathNum() -> compute pathway number of each level
%   logMatrix() -> compute channel connectivity matrix
%   rowPathRatio() -> compute horizontal propagation
%   logCorrCoef() -> compute vertical propagation
%   participateCoef() -> compute modified participation coefficient
%
%   Input:
%       network: a network structure.
%       maxLevel: max level for analyses
%       doNewNode: boolean for counting number of new nodes
%
%   Output:
%       network: a network structure with analyses result.

%   ---------
%   Yen-Nan Lin, NTHU, 2010-2014, Matlab 2012a

if nargin < 3
    doNewNode = false;
end

% reorder nodes with module
network = sortByModule(network);
network.NumModule = numel(unique(network.moduleId));

% compute number of new nodes within the pathway in different levels
if doNewNode == true
    [~, nNewNodeMat] = countIOPassNode(network, maxLevel);
    network.NumNewNodeMat = nNewNodeMat;
end

for iLevel = 1:maxLevel
    % compute pathway number of each level
    [levelMatrix, levelAllMatrix] = computePathNum(network, iLevel);
    network.LevelMatrix{iLevel} = single(full(levelMatrix));

    % build channel connectivity matrix
    network.ChConnectMat{iLevel} = logMatrix(network.LevelMatrix{iLevel});

    % compute horizontal propagation
    tmpHorzProp = rowPathRatio(network.LevelMatrix{iLevel});
    network.HorzProp(iLevel) = mean(tmpHorzProp);
end

% compute vertical propagation
network.VertProp = logCorrCoef(network);

% participation coefficient
network.PartpCoef = participateCoef(network);
end