function [ meanNetworks, allNetworks, stdNetworks ] = makeNetAndAnalysis( inputNetwork, parameter )
% MAKENETANDANALYSIS generate many kind of networks and do level analysis.
%
%   [ meanNetworks ] = makeNetAndAnalysis( inputNetwork, parameter )
%   This function will generate random network, ring type small world network,
%   distant input-output ring type small world network, ring type regular
%   network and use these networks do levelAnalysis.
%
%   Input:
%       inputNetwork: a input network structure
%       parameter: a structure containg parameter to analysis
%           parameter.MAX_LEVEL: max level of level analysis
%           parameter.REPEAT_TIMES: number of relization
%           parameter.STRIDE_SEED_NUMBER: stride between seed number
%           parameter.REWIRE_PROB_RING: rewiring probability of small world
%
%   Output:
%       meanNetworks: mean result of each kind of network
%       allNetworks: all relization networks
%       stdNetworks: std of each kinds of networks between relization

%   ---------
%   Yen-Nan Lin, NTHU, 2010-2014, Matlab 2012a

% parameter for analysis
loadStruct(parameter, 'caller');

% compute number of edges information of inputNetwork
inputNetwork.NumConnect = full(sum(sum(inputNetwork.matrix)));
inputNetwork.meanNumEdge = full(mean(sum(inputNetwork.matrix')));

% random network analysis
randProb = inputNetwork.NumConnect / (inputNetwork.size^2 - inputNetwork.size);
randNetwork = cell(1, REPEAT_TIMES);
for iTime = 1:REPEAT_TIMES
    seedNumber = iTime * STRIDE_SEED_NUMBER;
    randNetwork{iTime} = randomNetwork(inputNetwork.size, randProb, seedNumber);
    randNetwork{iTime} = setIO(randNetwork{iTime}, ...
        inputNetwork.inputNumber, inputNetwork.outputNumber);
    randNetwork{iTime} = levelAnalysis(randNetwork{iTime}, MAX_LEVEL);
end
% compute mean result of analysis
meanRandNet = meanFieldInCells(randNetwork);
stdRandNet = stdFieldInCells(randNetwork);
meanNetworks.meanRandNet = meanRandNet;
stdNetworks.stdRandNet = stdRandNet;
if nargout > 1
    allNetworks.randNetwork = randNetwork;
end


% ring type small world network analysis
smallRingNet = cell(1, REPEAT_TIMES);
for iTime = 1:REPEAT_TIMES
    seedNumber = iTime * STRIDE_SEED_NUMBER;
    smallRingNet{iTime} = smallWorldNetwork(inputNetwork.size, ...
        REWIRE_PROB_RING, seedNumber, inputNetwork.meanNumEdge);
    smallRingNet{iTime} = setIO(smallRingNet{iTime}, ...
        inputNetwork.inputNumber, inputNetwork.outputNumber);
    smallRingNet{iTime} = levelAnalysis(smallRingNet{iTime}, MAX_LEVEL);
end
% compute mean result of analysis
meanRingNet = meanFieldInCells(smallRingNet);
stdRingNet = stdFieldInCells(smallRingNet);
meanNetworks.meanRingNet = meanRingNet;
stdNetworks.stdRingNet = stdRingNet;
if nargout > 1
    allNetworks.smallRingNet = smallRingNet;
end


% distant IO with ring type small world network analysis
smallRingDistNet = cell(1, REPEAT_TIMES);
for iTime = 1:REPEAT_TIMES
    seedNumber = iTime * STRIDE_SEED_NUMBER;
    smallRingDistNet{iTime} = smallWorldNetwork(inputNetwork.size, ...
        REWIRE_PROB_RING, seedNumber, inputNetwork.meanNumEdge);
    smallRingDistNet{iTime} = setIO(smallRingDistNet{iTime}, ...
        inputNetwork.inputNumber, inputNetwork.outputNumber, 'distant');
    smallRingDistNet{iTime} = levelAnalysis(smallRingDistNet{iTime}, MAX_LEVEL);
end
% compute mean result of analysis
meanRingDistNet = meanFieldInCells(smallRingDistNet);
stdRingDistNet = stdFieldInCells(smallRingDistNet);
meanNetworks.meanRingDistNet = meanRingDistNet;
stdNetworks.stdRingDistNet = stdRingDistNet;
if nargout > 1
    allNetworks.smallRingDistNet = smallRingDistNet;
end

% ring type regular network analysis
regularRingNet = cell(1, REPEAT_TIMES);
for iTime = 1:REPEAT_TIMES
    seedNumber = iTime * STRIDE_SEED_NUMBER;
    regularRingNet{iTime} = smallWorldNetwork(inputNetwork.size, ...
        0, seedNumber, inputNetwork.meanNumEdge);
    regularRingNet{iTime} = setIO(regularRingNet{iTime}, ...
        inputNetwork.inputNumber, inputNetwork.outputNumber);
    regularRingNet{iTime} = levelAnalysis(regularRingNet{iTime}, MAX_LEVEL);
end
% compute mean result of analysis
meanRegularNet = meanFieldInCells(regularRingNet);
stdRegularNet = stdFieldInCells(regularRingNet);
meanNetworks.meanRegularNet = meanRegularNet;
stdNetworks.stdRegularNet = stdRegularNet;
if nargout > 1
    allNetworks.regularRingNet = regularRingNet;
end
end
