function [ meanValue ] = meanFieldInCells( cellArray, fieldName )
% MEANFIELDINCELLS compute mean field value of structures in cell arrays.
%
%   [ meanValue ] = meanFieldInCells( cellArray, fieldName )
%   meanFieldInCells() compute mean value of specific field in structure cell
%   array. If only one input argument exist, this function will compute all
%   numeric type fields in cellArray.
%
%   Input:
%       cellArray: structures in cell array
%       fieldName: name of field you want to compute
%
%   Output:
%       meanValue: mean value of specific field in cellArray. If you compute all
%           numeric field, meanValue will be a structure containing all mean of
%           numeric field in cellArray.

%   ---------
%   Yen-Nan Lin, NTHU, 2010-2014, Matlab 2012a

isAllField = false;
if nargin == 1
    isAllField = true;
end
nCell = length(cellArray);
if nCell == 0
    error('Input cell array is empty.');
end
if isAllField == true
    meanValue = [];
    allFieldName = fieldnames(cellArray{1});
    nField = length(allFieldName);
    for iField = 1:nField
        iFieldName = allFieldName{iField};
        iFieldValue = getfield(cellArray{1}, iFieldName);
        if ~isnumeric(iFieldValue)
            continue;
        end
        meanFieldValue = meanSpecificField(cellArray, iFieldName);
        meanValue = setfield(meanValue, iFieldName, meanFieldValue);
    end
else
    meanValue = meanSpecificField(cellArray, fieldName);
end
end

function [ meanValue ] = meanSpecificField( cellArray, fieldName )
nCell = length(cellArray);
sumField = getfield(cellArray{1}, fieldName);
if ~isnumeric(sumField)
    error([fieldName, ' is not numeric.']);
end
for iCell = 2:nCell
    if ~isfield(cellArray{iCell}, fieldName)
        error(['No ' fieldName 'field in ' int2str(iCell) 'th structure.']);
    end
    sumField = getfield(cellArray{iCell}, fieldName) + sumField;
end
meanValue = sumField / nCell;
end
