function [ network ] = setIO( network, inputNode, outputNode, selectType )
% SETIO set input and output node of network by selectType.
%
%    [ network ] = setIO( network, inputNode, outputNode, selectType )
%    If inputNode & outputNode are lists of nodes, than setIO() will set the
%    nodes in lists as input nodes and output nodes, respectively.
%    If inputNode & outputNode are numbers, than how to choose nodes as input
%    or output nodes is determined by selectType. If no selectType input
%    argument exist, it use 'random' type. Below info is about selectType.
%    random:
%        randomly select nodes as input or output nodes.
%    distant:
%        select nodes at one side of ring as input, select nodes at the other
%        side of ring as output, and the other nodes as inter nodes.
%
%    Input:
%        network: a network structure. When you want to use 'distant'
%            in selectType, the network should be ring type small world network.
%        inputNode: a number represent number of input nodes, and a vector
%            represent id list of input nodes.
%        outputNode: a number represent number of output nodes, and a vector
%            represent id list of output nodes.
%        selectType: the way to select nodes as input and output nodes.
%
%    Output:
%        network: a network structure with determined input and output nodes.
%
%    Example:
%        n = randomNetwork(9, 0.3, 0);
%        randIONetwork = setIO(n, 4, 2, 'random');
%        distantIONetwork = setIO(n, 2, 3, 'distant');
%

%   ---------
%   Yen-Nan Lin, NTHU, 2010-2014, Matlab 2012a

if nargin == 3
    selectType = 'random';
end

nNode = network.size;
% if inputNode and outputNode is scalar
if isscalar(inputNode) && isscalar(outputNode)
    nInput = inputNode;
    nOutput = outputNode;
    network.inputNumber = nInput;
    network.outputNumber = nOutput;
    switch selectType
        case 'random'
            ioList = randNoRepeat(nNode, nInput + nOutput);
        case 'distant'
            ioList = distantIOList(nNode, nInput, nOutput);
        otherwise
           error('Select type should be "random", "distant".');
    end
% if inputNode and outputNode is vector
elseif isvector(inputNode) && isvector(outputNode)
    nInput = length(inputNode);
    nOutput = length(outputNode);
    network.inputNumber = nInput;
    network.outputNumber = nOutput;
    % use find(..., 1) to prevent its size is (1, 1)
    dim = find( size(inputNode) ~= 1, 1);
    ioList = cat(dim, inputNode, outputNode);
else
    error('inputNode & outputNode should be scalar or vector');
end
network.inputList = ioList(1:nInput);
network.outputList = ioList((nInput + 1):end);
network = sortByIO(network, ioList);
end

function [ randList ] = randNoRepeat( randMax, nRand )
randList = randperm(randMax);
randList = randList(1:nRand);
end

function [ ioList ] = distantIOList( nNode, nInput, nOutput )
nInter = nNode - nInput - nOutput;
inputList = 1:nInput;
outputStartIx = nInput + floor(nInter / 2) + 1;
outputList = outputStartIx:(outputStartIx + nOutput - 1);
ioList = [inputList, outputList];
end

function [ network ] = sortByIO( network, ioList )
for ix = 1:length(ioList)
    wantNode = ioList(ix);
    if isnumeric(ioList)
        wantNodeIndex = find(network.label == wantNode, 1);
    elseif ischar(ioList)
        error('Still cannot access string label');
    end
    network.matrix = swapCol(network.matrix, ix, wantNodeIndex);
    network.matrix = swapRow(network.matrix, ix, wantNodeIndex);
    network.label = swapCol(network.label, ix, wantNodeIndex);
end
end

function [ matrix ] = swapCol( matrix, col1, col2 )
matrix(:, [col1, col2]) = matrix(:, [col2, col1]);
end

function [ matrix ] = swapRow( matrix, col1, col2 )
matrix([col1, col2], :) = matrix([col2, col1], :);
end
