function [ network ] = smallHubNetwork( nNode, rewireProb, seedNumber, nEdge, nHub, hubStyle )
% SMALLHUBNETWORK create Watts & Strogatz small world network with hubs.
%
%   [ network ] = smallHubNetwork( nNode, rewireProb, seedNumber, nEdge, nHub, hubStyle )
%   This function not only create Watts & Strogatz small world network but also
%   replace nHub nodes by hubs (provincial, connect or kinless determined by
%   hubStyle). The degrees of hubs are determined from tail of exponetial
%   distribution (its mean is nEdge - 1). Remaining degrees would be uniformly
%   and randomly distributed to each non-hub nodes.
%   The input argument hubStyle control the connection mechanism of hubs.
%   3 different mechanisms would let most of hubs be classifed into the specific
%   kinds of hubs. Below is the mechanism:
%   Provincial: 100% connections link with neighbour nodes of ring lattice.
%   Kinless: 100% connections link with random nodes.
%   Connector: 80% connections link with neighbours and 20% connections link
%       with random nodes.
%
%   Input:
%       nNode: number of node of this network
%       rewireProb: rewiring rewireProb
%       seedNumber: seed number of random
%       nEdge: number of edge of each node
%       nHub: number of hubs
%       hubStyle: provincial, connector or kinless
%
%    Output:
%        network: a network structure

%   ---------
%   Yen-Nan Lin, NTHU, 2010-2014, Matlab 2012a

RATIO_RAND = 0.20;
if nargin < 4
    error('No sufficient input arguments.')
end

network = struct('size', nNode, 'rewireProb', rewireProb, ...
    'seed', seedNumber, 'nEdge', nEdge);
rand('seed', seedNumber);
network.matrix = sparse(nNode, nNode);

% classify hub types
isProvincial = false;
isConnector = false;
isKinless = false;
if strcmpi(hubStyle, 'provincial') == true
    isProvincial = true;
elseif strcmpi(hubStyle, 'connector') == true
    isConnector = true;
elseif strcmpi(hubStyle, 'kinless') == true
    isKinless = true;
else
    error(['no ' hubStyle ' this kind of hubStyle.']);
end

% assign degree to each nodes
nNonHub = nNode - nHub;
outDegList = round(exprnd(nEdge - 1, 1, nNode)) + 1;
[tmpSort, tmpSortIx] = sort(outDegList, 'descend');
hubIxList = tmpSortIx(1:nHub);
nonHubIxList = tmpSortIx((nHub + 1):end);
remainSumDeg = nNode * nEdge - sum(outDegList(hubIxList));
nEdgeNonHub = floor(remainSumDeg / nNonHub);
outDegList(nonHubIxList) = nEdgeNonHub;
if mod(remainSumDeg, nNonHub) > 0
    tmpRandIx = randperm(nNonHub);
    tmpRandIx = tmpRandIx(1:mod(remainSumDeg, nNonHub));
    tmpRandIx = tmpRandIx + nHub;
    outDegList(tmpSortIx(tmpRandIx)) = outDegList(tmpSortIx(tmpRandIx)) + 1;
end

% sort hub to last entry of outDegList
if isKinless == true
    outDegList = horzcat(outDegList, outDegList(hubIxList));
    outDegList(hubIxList) = [];
    [tmpSort, tmpSortIx] = sort(outDegList, 'descend');
    hubIxList = (nNode - nHub + 1):nNode;
    nonHubIxList = 1:(nNode - nHub);
end

% build connection
for iNode = 1:numel(tmpSortIx)
    nodeIx = tmpSortIx(iNode);

    isHub = false;
    if ~isempty(find(hubIxList == nodeIx, 1))
        isHub = true;
    end

    % if already exist some link to hub
    tmpExistOutLink = find(network.matrix(nodeIx, :) > 0);
    if ~isempty(tmpExistOutLink)
        tmpOutDeg = outDegList(nodeIx) - numel(tmpExistOutLink);
    else
        tmpOutDeg = outDegList(nodeIx);
    end

    % part of connector hub link to neighbor, the other link to random nodes
    if isConnector == true && isHub == true
        tmpRandOutDeg = round(tmpOutDeg * RATIO_RAND);
        tmpOutDeg = tmpOutDeg - tmpRandOutDeg;
    end

    % create array of relative neighbor index
    radius = floor(tmpOutDeg / 2);
    neighbor = [-radius:-1, 1:radius];
    if mod(tmpOutDeg, 2) == 1
        if rand() > 0.5
            neighbor = [neighbor, radius + 1];
        else
            neighbor = [-(radius + 1), neighbor];
        end
    end

    % kinless hub don't link to neighbor
    if isKinless == true && isHub == true
        neighbor = [];
    end

    % link to neighbor
    for jNeighbor = 1:numel(neighbor)
        connectIx = nodeIx + neighbor(jNeighbor);

        if isKinless == true
            connectIx = ringIndex(connectIx, nNode - nHub);
        else
            connectIx = ringIndex(connectIx, nNode);
        end

        while network.matrix(nodeIx, connectIx) == 1
            if neighbor(jNeighbor) > 0
                neighbor(jNeighbor) = neighbor(jNeighbor) + 1;
            else
                neighbor(jNeighbor) = neighbor(jNeighbor) - 1;
            end

            connectIx = nodeIx + neighbor(jNeighbor);
            if isKinless == true
                connectIx = ringIndex(connectIx, nNode - nHub);
            else
                connectIx = ringIndex(connectIx, nNode);
            end
        end

        network.matrix(nodeIx, connectIx) = 1;
        % create reciprocal link due to high in and out degree of hub
        if isHub == true
            network.matrix(connectIx, nodeIx) = 1;
        end
    end

    if (isKinless == true || isConnector == true) && isHub == true
        tmpExistOutLink = find(network.matrix(nodeIx, :) > 0);
        tmpOutDeg = outDegList(nodeIx) - numel(tmpExistOutLink);
        tmpExistInLink = find(network.matrix(:, nodeIx) > 0);
        tmpInDeg = outDegList(nodeIx) - numel(tmpExistInLink);

        % link to random nodes
        tmpValidIx = 1:nNode;
        tmpValidIx([tmpExistOutLink, nodeIx, tmpSortIx(1:iNode)]) = [];
        tmpValidIx = tmpValidIx(randperm(numel(tmpValidIx)));
        tmpValidIx = tmpValidIx(1:tmpOutDeg);
        network.matrix(nodeIx, tmpValidIx) = 1;

        % link from random nodes
        tmpValidIx = 1:nNode;
        tmpValidIx([tmpExistInLink; nodeIx; tmpSortIx(1:iNode)']) = [];
        tmpValidIx = tmpValidIx(randperm(numel(tmpValidIx)));
        tmpValidIx = tmpValidIx(1:tmpInDeg);
        network.matrix(tmpValidIx, nodeIx) = 1;
    end
end

% check the node with in/out degree = 0 exist or not
tmpConnectMat = randomRewire(rewireProb, network.matrix, nNode, hubIxList);
while(~isempty(find(sum(tmpConnectMat, 1) == 0, 1)) || ...
        ~isempty(find(sum(tmpConnectMat, 2) == 0, 1)))
    disp('Exist a node with 0 degree');
    tmpConnectMat = ...
        randomRewire(rewireProb, network.matrix, nNode, hubIxList);
end
network.matrix = tmpConnectMat;
network.label = 1:nNode;
network.HubLabel = hubIxList;
end

function [ matrix ] = randomRewire( rewireProb, matrix, nNode, hubIxList )
% check neurons can be rewired or not
if ~isempty(find(sum(matrix, 1) > (nNode - 1), 1))
    error({'Exist a neuron which connect all other neurons.', ...
        'Cannot rewire its connection.'});
end

[rowIxList, colIxList] = find(matrix > 0);
for iConnect = 1:numel(rowIxList)
    if rand() > rewireProb
        continue;
    end

    rowIx = rowIxList(iConnect);
    colIx = colIxList(iConnect);

    % don't rewire hub's connections
    if ~isempty(find(hubIxList == rowIx, 1)) || ...
            ~isempty(find(hubIxList == colIx, 1))
        continue;
    end

    % change source node
    if rand() > 0.5
        newRowIx = randi(nNode);
        newColIx = colIx;
        while matrix(newRowIx, newColIx) == 1 || newRowIx == newColIx
            newRowIx = randi(nNode);
        end
    % change destination node
    else
        newRowIx = rowIx;
        newColIx = randi(nNode);
        while matrix(newRowIx, newColIx) == 1 || newRowIx == newColIx
            newColIx = randi(nNode);
        end
    end

    matrix(rowIx, colIx) = 0;
    matrix(newRowIx, newColIx) = 1;
end
end

function [ ringIx ] = ringIndex( ix, nNode )
while ix > nNode
    ix = ix - nNode;
end
while ix <= 0
    ix = ix + nNode;
end
ringIx = ix;
end
