function [ network ] = smallWorldNetwork( nNode, probability, seedNumber, kEdge )
% SMALLWORLDNETWORK create Watts & Strogatz small world network.
%
%   [ network ] = smallWorldNetwork( nNode, probability, seedNumber, kEdge )
%   This function create Watts & Strogatz small world network. If the kEdge is
%   odd, it will randomly connect one side. If the kEdge contain fraction,
%   remaining edges would be randomly distributed in the network.
%
%   Input:
%       nNode: number of node of this network
%       probability: rewiring probability
%       seedNumber: seed number of random
%       kEdge: number of edge of each node
%
%   Output:
%       network: a network structure

%   ---------
%   Yen-Nan Lin, NTHU, 2010-2014, Matlab 2012a

if nargin < 4
    error('No sufficient input arguments.')
end

network = struct('size', nNode, 'rewireProb', probability, ...
    'seed', seedNumber, 'kEdge', kEdge);
rand('seed', seedNumber);
network.matrix = sparse(nNode, nNode);

% assign degree to each nodes
outDegList = ones(1, nNode) * floor(kEdge);
remainSumDeg = round(nNode * kEdge) - sum(outDegList);
if mod(remainSumDeg, nNode) > 0
    tmpRandIx = randperm(nNode);
    tmpRandIx = tmpRandIx(1:mod(remainSumDeg, nNode));
    outDegList(tmpRandIx) = outDegList(tmpRandIx) + 1;
end

% build connection
for iNode = 1:nNode
    tmpOutDeg = outDegList(iNode);

    % create array of relative neighbor index
    radius = floor(tmpOutDeg / 2);
    neighbor = [-radius:-1, 1:radius];
    if mod(tmpOutDeg, 2) == 1
        if rand() > 0.5
            neighbor = [neighbor, radius + 1];
        else
            neighbor = [-(radius + 1), neighbor];
        end
    end

    for jNeighbor = 1:numel(neighbor)
        toNodeIx = iNode + neighbor(jNeighbor);
        toNodeIx = ringIndex(toNodeIx, nNode);
        network.matrix(iNode, toNodeIx) = 1;
    end
end

% check the node with in/out degree = 0 exist or not
tmpConnectMat = randomRewire(network.rewireProb, network.matrix, network.size);
while(~isempty(find(sum(tmpConnectMat, 1) == 0, 1)) || ...
        ~isempty(find(sum(tmpConnectMat, 2) == 0, 1)))
    disp('Exist a node with 0 degree');
    tmpConnectMat = ...
        randomRewire(network.rewireProb, network.matrix, network.size);
end
network.matrix = tmpConnectMat;
network.label = 1:nNode;
end

function [ matrix ] = randomRewire( rewireProb, matrix, nNode )
% check neurons can be rewired or not
if ~isempty(find(sum(matrix, 1) > (nNode - 1), 1))
    error({'Exist a neuron which connect all other neurons.', ...
        'Cannot rewire its connection.'});
end

[rowIxList, colIxList] = find(matrix > 0);
for iConnect = 1:numel(rowIxList)
    if rand() > rewireProb
        continue;
    end

    rowIx = rowIxList(iConnect);
    colIx = colIxList(iConnect);

    % change source node
    if rand() > 0.5
        newRowIx = randi(nNode);
        newColIx = colIx;
        while matrix(newRowIx, newColIx) == 1 || newRowIx == newColIx
            newRowIx = randi(nNode);
        end
    % change destination node
    else
        newRowIx = rowIx;
        newColIx = randi(nNode);
        while matrix(newRowIx, newColIx) == 1 || newRowIx == newColIx
            newColIx = randi(nNode);
        end
    end

    matrix(rowIx, colIx) = 0;
    matrix(newRowIx, newColIx) = 1;
end
end

function [ ringIx ] = ringIndex( ix, nNode )
while ix > nNode
    ix = ix - nNode;
end
while ix <= 0
    ix = ix + nNode;
end
ringIx = ix;
end
