function [ stdCell ] = stdFieldInCells( cellArray, fieldName )
% MEANFIELDINCELLS compute std field value of structures in cell arrays.
%
%   [ stdCell ] = stdFieldInCells( cellArray, fieldName )
%   stdFieldInCells() compute std value of specific field in structure cell
%   array. If only one input argument exist, this function will compute all
%   numeric type fields in cellArray.
%
%   Input:
%       cellArray: structures in cell array
%       fieldName: name of field you want to compute
%
%   Output:
%       stdCell: std value of specific field in cellArray. If you compute all
%           numeric field, stdCell will be a structure containing all std of
%           numeric field in cellArray.

%   ---------
%   Yen-Nan Lin, NTHU, 2010-2014, Matlab 2012a

isAllField = false;
if nargin == 1
    isAllField = true;
end
nCell = length(cellArray);
if nCell == 0
    error('Input cell array is empty.');
end

if isAllField == true
    stdCell = [];
    allFieldName = fieldnames(cellArray{1});
    nField = length(allFieldName);
    % compute std value of each field
    for iField = 1:nField
        iFieldName = allFieldName{iField};
        iFieldValue = getfield(cellArray{1}, iFieldName);
        if ~isnumeric(iFieldValue)
            continue;
        end
        stdFieldValue = stdSpecificField(cellArray, iFieldName);
        stdCell = setfield(stdCell, iFieldName, stdFieldValue);
    end
else
    stdCell = stdSpecificField(cellArray, fieldName);
end
end

function [ stdFieldValue ] = stdSpecificField( cellArray, fieldName )
% compute std value of specific field in cell array

nCell = length(cellArray);
fieldValue = full(getfield(cellArray{1}, fieldName));
if ~isnumeric(fieldValue)
    error([fieldName, ' is not numeric.']);
end
catDimIx = find(size(fieldValue) == 1, 1);
if isempty(catDimIx)
    catDimIx = numel(size(fieldValue)) + 1;
end

% Concatenate all value of specific field in cell array
for iCell = 2:nCell
    if ~isfield(cellArray{iCell}, fieldName)
        error(['No ' fieldName 'field in ' int2str(iCell) 'th structure.']);
    end
    fieldValue = ...
        cat(catDimIx, fieldValue, full(getfield(cellArray{iCell}, fieldName)));
end

stdFieldValue = std(fieldValue, 0, catDimIx);
end
